from django.views import View
from django.http import JsonResponse
import json
import os
from users.models import SecureUser, LoginActivity, FailedLoginAttempt
from users.core import HashMaster, SecretVault, TokenForge
from django.views.decorators.csrf import csrf_exempt
from django.utils.decorators import method_decorator
from django.utils import timezone
from datetime import timedelta
from employer_profile.models import Profile
from django.conf import settings



@method_decorator(csrf_exempt, name='dispatch')
class UserRegistration(View):
    def post(self, request):
        if request.content_type != 'application/json':
            return JsonResponse({'error': 'Invalid content type'}, status=400)

        try:
            body = request.body
            if not body:
                return JsonResponse({'error': 'Empty request body'}, status=400)
            
            data = json.loads(body)
            email = data.get('email')
            password = data.get('password')

            if not email or not password:
                return JsonResponse({'error': 'Email and password are required'}, status=400)

            # Use exists() with a .only() to reduce DB load
            if SecureUser.objects.only('email').filter(email=email).exists():
                return JsonResponse({'error': 'Email already registered'}, status=400)

            # Reuse salt and hasher efficiently
            salt = os.urandom(32)
            hasher = HashMaster(SecretVault())
            password_hash = hasher.create_password_digest(password, salt)

            SecureUser.objects.create(
                email=email,
                password_hash=password_hash,
                salt=salt
            )

            return JsonResponse({'message': 'User created successfully'}, status=201)

        except json.JSONDecodeError:
            return JsonResponse({'error': 'Invalid JSON'}, status=400)
        except Exception as e:
            return JsonResponse({'error': str(e)}, status=500)

@method_decorator(csrf_exempt, name='dispatch')
class UserLogin(View):
    def post(self, request):
        try:
            # Parse request data
            try:
                data = json.loads(request.body)
                email = data.get('email', '').strip()
                password = data.get('password', '').strip()
                
                if not email or not password:
                    raise ValueError('Email and password are required')
            except json.JSONDecodeError:
                return JsonResponse(
                    {'error': 'Invalid JSON format'},
                    status=400
                )
            except ValueError as e:
                return JsonResponse(
                    {'error': str(e)},
                    status=400
                )

            print(f"Login attempt for: {email}")

            try:
                user = SecureUser.objects.get(email=email)
            except SecureUser.DoesNotExist:
                return JsonResponse(
                    {'error': 'Invalid credentials'},
                    status=401
                )

            # Account lock check
            if user.account_locked_until and user.account_locked_until > timezone.now():
                remaining_time = (user.account_locked_until - timezone.now()).seconds // 60
                return JsonResponse(
                    {
                        'error': f'Account temporarily locked. Try again in {remaining_time} minutes',
                        'locked_until': user.account_locked_until.isoformat()
                    },
                    status=403
                )

            # Password verification
            hasher = HashMaster(SecretVault())
            if not hasher.verify_password(password, user.password_hash):
                user.failed_login_attempts += 1
                
                # Lock account after 5 failed attempts
                if user.failed_login_attempts >= 5:
                    user.account_locked_until = timezone.now() + timedelta(minutes=30)
                    user.save()
                    return JsonResponse(
                        {
                            'error': 'Too many failed attempts. Account locked for 30 minutes.',
                            'locked_until': user.account_locked_until.isoformat()
                        },
                        status=403
                    )
                
                user.save()
                return JsonResponse(
                    {'error': 'Invalid credentials'},
                    status=401
                )

            # Successful login - reset counters
            user.failed_login_attempts = 0
            user.last_login = timezone.now()
            user.save()

            # Get profile data if exists
            try:
                # Get the most recent profile if multiple exist
                profile = Profile.objects.filter(users=user).order_by('-created_at').first()
                if profile:
                    profile_data = {
                        'id': profile.id,
                        'name': profile.name,
                        'created_at': profile.created_at.isoformat(),
                    }
                else:
                    profile_data = None
            except Exception as e:
                print(f"Error fetching profile: {str(e)}")
                profile_data = None

            # Generate token
            token = TokenForge(SecretVault()).craft_token(
                user.id,
                {
                    'role': 'admin' if user.is_admin else 'user',
                    'email': user.email
                }
            )

            # Prepare response data
            response_data = {
                'token': token,
                'user': {
                    'id': user.id,
                    'email': user.email,
                    'is_admin': user.is_admin,
                    'last_login': user.last_login.isoformat() if user.last_login else None
                },
                'profile': profile_data
            }
            print(response_data, "response data")   
            return JsonResponse(response_data)

        except Exception as e:
            # Log the full error for debugging
            print(f"Login error: {str(e)}")  # Add detailed error logging
            return JsonResponse(
                {
                    'error': 'An unexpected error occurred during authentication',
                    'details': str(e) if settings.DEBUG else None  # Include error details only in debug mode
                },
                status=500
            )
    
@method_decorator(csrf_exempt, name='dispatch')
class RecentActivities(View):
    def get(self, request):
        try:
            # Get Authorization header (more reliable)
            auth_header = request.META.get('HTTP_AUTHORIZATION', '')
            if not auth_header.startswith('Bearer '):
                return JsonResponse({'error': 'Authorization header missing or malformed'}, status=401)
            
            token = auth_header[7:]  # Remove 'Bearer ' prefix
            if not token:
                return JsonResponse({'error': 'No token provided'}, status=401)
            
            # Decode token
            token_data, valid = TokenForge(SecretVault()).decode_token(token)
            if not valid:
                return JsonResponse({'error': 'Invalid token'}, status=401)
            
            try:
                user = SecureUser.objects.get(id=token_data['sub'])
            except SecureUser.DoesNotExist:
                return JsonResponse({'error': 'User not found'}, status=401)
            
            # Fetch login activities
            login_activities = LoginActivity.objects.filter(user=user).order_by('-timestamp')[:10]
            # Fetch failed login attempts
            failed_attempts = FailedLoginAttempt.objects.filter(user=user).order_by('-timestamp')[:10]
            
            activities = []
            
            for activity in login_activities:
                activities.append({
                    'type': 'login',
                    'status': activity.status,
                    'timestamp': activity.timestamp.isoformat(),
                    'ip_address': activity.ip_address,
                    'user_agent': activity.user_agent,
                })
            
            for attempt in failed_attempts:
                activities.append({
                    'type': 'failed_login',
                    'timestamp': attempt.timestamp.isoformat(),
                    'ip_address': attempt.ip_address,
                    'user_agent': attempt.user_agent,
                })
            
            # Sort by timestamp descending
            activities.sort(key=lambda x: x['timestamp'], reverse=True)
            
            return JsonResponse({'activities': activities[:20]})
        
        except Exception as e:
            # Log error for debugging
            print(f"Error fetching activities: {str(e)}")
            return JsonResponse(
                {
                    'error': 'An error occurred while fetching activities',
                    'details': str(e) if settings.DEBUG else None,
                },
                status=500
            )

           
@method_decorator(csrf_exempt, name='dispatch')
class AdminUserLogin(View):
    def post(self, request):
        try:
            data = json.loads(request.body)
            email = data['email']
            password = data['password']
            
            print(f"Admin login attempt for: {email}")  # Debug
            
            user = SecureUser.objects.get(is_admin=True, email=email)
            
            # Debug prints
            print(f"Stored hash: {user.password_hash}")
            print(f"Input password: {password}")    
            
            # Account lock check
            if user.account_locked_until and user.account_locked_until > timezone.now():
                return JsonResponse(
                    {'error': 'Account temporarily locked'},
                    status=403
                )
                
            hasher = HashMaster(SecretVault())
            
            if not hasher.verify_password(password, user.password_hash):
                user.failed_login_attempts += 1
                print(f"Failed attempts: {user.failed_login_attempts}")  # Debug
                
                if user.failed_login_attempts >= 5:
                    user.account_locked_until = timezone.now() + timedelta(minutes=30)
                    user.save()
                    return JsonResponse(
                        {'error': 'Too many failed attempts. Account locked for 30 minutes.'},
                        status=403
                    )
                    
                user.save()
                return JsonResponse(
                    {'error': 'Invalid credentials'},
                    status=401
                )
            
            # Successful login
            user.failed_login_attempts = 0
            user.last_login = timezone.now()
            user.save()

            # Get profile data if exists
            try:
                # Get the most recent profile if multiple exist
                profile = Profile.objects.filter(users=user).order_by('-created_at').first()
                if profile:
                    profile_data = {
                        'id': profile.id,
                        'name': profile.name,
                        'created_at': profile.created_at.isoformat(),
                    }
                else:
                    profile_data = None
            except Exception as e:
                print(f"Error fetching profile: {str(e)}")
                profile_data = None
            
            token = TokenForge(SecretVault()).craft_token(
                user.id,
                {'role': 'admin'}
            )
            
            return JsonResponse({
                'token': token,
                'user_id': user.id,
                'email': user.email,
                'profile': profile_data
            })
            
        except SecureUser.DoesNotExist:
            print(f"Admin user not found: {email}")  # Debug
            return JsonResponse(
                {'error': 'Invalid credentials'},
                status=401
            )
        except Exception as e:
            print(f"Admin login error: {str(e)}")  # Debug
            return JsonResponse(
                {'error': 'Authentication failed'},
                status=400
            )



@method_decorator(csrf_exempt, name='dispatch')
class GetUsers(View):
    def get(self, request):
        try:
            # Get all users from database
            users = SecureUser.objects.all()
            
            # Create list of user data
            user_list = []
            for user in users:
                user_data = {
                    'id': user.id,
                    
                    'email': user.email,
                    'last_login': user.last_login,
                    'failed_login_attempts': user.failed_login_attempts,
                    'account_locked_until': user.account_locked_until
                }
                user_list.append(user_data)
            
            return JsonResponse({
                'users': user_list
            }, status=200)
            
        except Exception as e:
            return JsonResponse({
                'error': str(e)
            }, status=500)


@method_decorator(csrf_exempt, name='dispatch')
class PasswordReset(View):
    def post(self, request):
        data = json.loads(request.body)
        email = data.get("email")
        new_password = data.get("password")

        if not email or not new_password:
            return JsonResponse({"message": "Email and password required"}, status=400)

        try:
            user = SecureUser.objects.get(email=email)
        except SecureUser.DoesNotExist:
            return JsonResponse({"message": "User not found"}, status=404)

        # Hash and store
        salt = os.urandom(32)
        hasher = HashMaster(SecretVault())
        password_hash = hasher.create_password_digest(new_password, salt)

        user.salt = salt
        user.password_hash = password_hash
        user.save()


        return JsonResponse({"message": "Password updated successfully"})

#TODO
class PasswordChange(View):
    def post(self, request):
        # Implementation for password change flow
        pass