import hashlib
import hmac
import os
import base64
import time
import json
from typing import Tuple

class SecretVault:
    def __init__(self):
        self.pepper = self._generate_pepper()
        
    def _generate_pepper(self) -> bytes:
        return os.urandom(64)
    
    def get_pepper(self) -> bytes:
        return b'supersecretpepper'  

class HashMaster:
    def __init__(self, vault: SecretVault):
        self.vault = vault
        self.iterations = 250000

    def create_password_digest(self, password: str, salt: bytes) -> str:
        password = password.strip()
        pepper = self.vault.get_pepper()
        dk = hashlib.pbkdf2_hmac('sha512', password.encode('utf-8') + pepper, salt, self.iterations)
        return f"sha512${self.iterations}${salt.hex()}${dk.hex()}"

    def verify_password(self, password: str, stored_hash: str) -> bool:
        try:
            algorithm, iterations, salt_hex, hash_hex = stored_hash.split('$')
            salt = bytes.fromhex(salt_hex)
            pepper = self.vault.get_pepper()
            dk = hashlib.pbkdf2_hmac('sha512', password.strip().encode('utf-8') + pepper, salt, int(iterations))
            new_hash = f"{algorithm}${iterations}${salt_hex}${dk.hex()}"
            print("Expected:", stored_hash)
            print("Generated:", new_hash)
            return hmac.compare_digest(new_hash, stored_hash)
        except Exception as e:
            print("Error:", e)
            return False

class TokenForge:
    def __init__(self, vault: SecretVault):
        self.vault = vault
        
    def craft_token(self, user_id: int, payload: dict) -> str:
        """Create signed JWT token"""
        header = {
            "alg": "HS512",
            "typ": "JWT"
        }
        
        current_time = int(time.time())
        payload.update({
            "sub": user_id,
            "iat": current_time,
            "exp": current_time + 3600,  # 1 hour expiration
            "iss": "secure_api_gate",
            "aud": "brilliant_user_base"
        })
        
        encoded_header = base64.urlsafe_b64encode(json.dumps(header).encode()).decode().rstrip('=')
        encoded_payload = base64.urlsafe_b64encode(json.dumps(payload).encode()).decode().rstrip('=')
        
        signature = hmac.new(
            self.vault.get_pepper(),
            f"{encoded_header}.{encoded_payload}".encode(),
            hashlib.sha512
        ).digest()
        
        encoded_signature = base64.urlsafe_b64encode(signature).decode().rstrip('=')
        
        return f"{encoded_header}.{encoded_payload}.{encoded_signature}"
    
    def decode_token(self, token: str) -> Tuple[dict, bool]:
        """Verify and decode JWT token"""
        try:
            encoded_header, encoded_payload, encoded_signature = token.split('.')
            
            # Add padding back if needed
            payload_padding = len(encoded_payload) % 4
            if payload_padding:
                encoded_payload += '=' * (4 - payload_padding)
            
            signature_check = hmac.new(
                self.vault.get_pepper(),
                f"{encoded_header}.{encoded_payload}".encode(),
                hashlib.sha512
            ).digest()
            
            if not hmac.compare_digest(
                base64.urlsafe_b64decode(encoded_signature + '=' * (4 - len(encoded_signature) % 4)),
                signature_check
            ):
                return {}, False
                
            payload = json.loads(base64.urlsafe_b64decode(encoded_payload))
            return payload, True
        except:
            return {}, False